{$I FD_incl.pas}
unit fd_24C65;

{* Fr den 'neuen' 'grossen' EEPROM auf der Huckepack-Platine.
 * Ansonsten siehe FD_PROM
 *}

(******************************************************************)
(* This Unit is copyrighted by Werner Cornelius DG3DBI.          *)
(* It may only be distributed with this Copyright notice and the  *)
(* following explanations for non-commercial amateur radio use    *)
(* with FALCon-25 Controller.                                     *)
(* Unit zur Ansteuerung des seriellen EEPROMs 24C65 auf FALCon-25 *)
(* fr Digiware. Dieser EEPROM-Typ hat eine Kapazitt von 8KByte. *)
(* Die Unit besitzt eine Autodetect-Funktion fr das EEPROM, die  *)
(* beim ffnen des EEPROM-Textfiles im Fehlerfall, also nicht     *)
(* existendem EEPROM einen Fehler (Path not found (3)) liefert.   *)
(* Ein Fehlercode von 32 dez. zeigt einen doppelten Zugriff an.   *)
(* Zur Verwaltung des EEPROMs lsst sich ein ganz normal global   *)
(* oder besser lokal oder dynamisch definiertes Textfile benutzen.*)
(* Dieses Textfile mu vor einem reset oder rewrite erst durch ein*)
(* spezielles assign24c65(eepromfile) behandelt werden.           *)
(* Ein normales Pascal assign ist nicht verwendbar !              *)
(* Whrend eines Schreibvorganges wird die gesamte Programmierzeit*)
(* ohne Bettigung einer Watchdog gewartet. (pro 8 Byte ca. 10ms) *)
(* Die Variable p0mirror ist die Spiegelvariable fr das nicht    *)
(* rcklesbare Register p0m des V25. Alle Module, die dieses Re-  *)
(* gister verndern sollten nderungen des Zustandes auch in der  *)
(* Variablen anzeigen, damit keine undefinierten Zustnde auftre- *)
(* ten.                                                           *)
(* Diese Unit benutzt mit Absicht keine anderen Units! (..Es war einmal) *)
(******************************************************************)
interface

{$IFDEF scc}
  uses fd_tnc ; {* wg. pm0mirror *}
{$ELSE}
  uses fd_crt; {* wg. watchdog *}
{$ENDIF}


(************************************************************************)
(* Funktion zum Setzen eines Standard-Textfiles fr Benutzung mit       *)
(* dem eeprom. Diese Prozedur mu auf jeden Fall VOR dem Aufruf eines   *)
(* Rewrite oder Resets durchgefhrt werden.                             *)
(************************************************************************)
procedure assign24c65(var eefile:text);
(* Hier beginnt der Programmfrickel *)
(* 1914 Bytes Code, 4 Byte Daten    *)

implementation
const
  EEPROM_SIZE = 8192;
  SDA_BIT     = $01;           (* Datenbit V25 serielle Ausgabe *)
  SCL_BIT     = $02;           (* Clockbit serielle Ausgabe *)

  (* Definitionen fr das Feld Mode im Textrec *)
  fmClosed = $D7B0;
  fmInput  = $D7B1;
  fmOutput = $D7B2;

type
  (* Definition des Borland Text-Records *)
  PTextBuf = ^TextBuf;
  TextBuf  = array[0..127] of Char;
  BigBuffer= array[0..EEPROM_SIZE-1] of byte;
  TextRec  = record
               Handle:    Word;
               Mode:      Word;
               BufSize:   Word;
               Private:   Word;
               BufPos:    Word;
               BufEnd:    Word;
               BufPtr:    PTextBuf;
               OpenFunc:  Pointer;
               InOutFunc: Pointer;
               FlushFunc: Pointer;
               CloseFunc: Pointer;
            (* Die hier eingeschlossenen Vars sind durch Krzen von userdata
               eingefgt worden. *)
               bufferakt: word;      (* Anzahl Bytes R/W Puffer/EEPROM *)
               read_eof : boolean;   (* EOF bei Read erreicht *)
               buffpruef: byte;      (* Prfsumme Buffer *)
            (* Ende Einfgung *)
               UserData:  array[1..16-4] of Byte;
               Name:      array[0..79] of Char;
               Buffer:    TextBuf;
             end;
  out_type = record
               cmd_bus : byte; (* Kommando/Adresse *)
               adresse : word; (* Adresse High/Low *)
               expand  : byte; (* weiteres Kommandobyte *)
             end;


var
  eeprom_inuse : boolean; (* EEPROM wird schon benutzt *)
  eeadr        : byte;    (* Adresse des EEPROMs auf dem IIC-Bus *)

(*$IFnDEF SCC *)
  filePC          : file;
(*$ENDIF *)


(*******************************************)
(* Ausgabe auf den IIC-Bus. Ggf. starten   *)
(* und folgende anz Bytes ausgeben         *)
(* ist nostop gesetzt, wird keine Stop-    *)
(* bedingung vor dem Zugriff erzeugt.      *)
(* Ist nostart gesetzt wird keine Start-   *)
(* bedingung erzeugt und direkt ausgegeben.*)
(* Registerbelegung:                       *)
(* ds    : Zeiger auf V25-IO-Segment       *)
(* di    : Zeiger Port P0 (Daten)          *)
(* ch,cl : High und Low Werte SCL          *)
(* dh,dl : High und Low Werte SDA          *)
(* es:bx : Zeiger auf Datenbytes           *)
(* ah    : Bytezhler                      *)
(* al    : temporre Bitdaten              *)
(* si    : Loopzhler                      *)
(*******************************************)
function out_iic(var buf;anz:byte;nostop,nostart:boolean):integer;
begin (* out_iic *)
  out_iic := 0; (* falls keine Lnge oder kein Fehler *)
(*$IFDEF SCC *)
  asm   (* wegen Geschwindigkeit wird die Operation in asm ausgefhrt *)
           les   bx,dword ptr [buf]           (* Zeiger Ausgabe *)
           pushf                              (* Interruptzustand *)
           cli                                (* retten + IRQ sperren *)
           mov   dh,[PM0_mirror]                (* Output Variable P0 Direction *)
           push  ds                           (* fr Registerzugriff retten *)
           mov   ax,seg p0                    (* Segment V25-IO *)
           mov   ds,ax                        (* nach ds *)
           and   dh,not SCL_BIT               (* SCL immer Output *)
           or    dh,SDA_BIT                   (* SDA als Input *)
           mov   dl,dh                        (* nach dl *)
           and   dl,not SDA_BIT               (* Ausgang ein SDA = 0 *)
           mov   di,offset P0                 (* Zeiger Daten 0 *)
           mov   ch,[di]                      (* aktuellen Zustand lesen *)
           and   ch,not SDA_BIT               (* Data auf 0 *)
           or    ch,SCL_BIT                   (* Clock auf 1 *)
           mov   cl,ch                        (* Clock Low *)
           and   cl,not SCL_BIT               (* setzen *)
           mov   ah,[anz]                     (* Anzahl *)
           test  [nostart],0FFh               (* kein Start erzeugen ? *)
           jnz   @@nstart                     (* genau *)
           test  [nostop],0FFh                (* kein Stop erzeugen ? *)
           jnz   @@nstop                      (* genau *)
           mov   [di],ch                      (* Ausgaberegister Daten mit SCL high *)
           mov   [di+1],dl                    (* Ausgang Daten auf Low *)
           nop
           nop
           nop
           nop
           nop
           nop
           mov   [di+1],dh                    (* Stop generieren *)
           nop
           nop
           nop
           nop
           nop
           nop
           (* Die Hardware ist nun vorbereitet *)
           (* Die Startbedingung generieren    *)
@@nstop:   mov   [di+1],dh                    (* Stop generieren *)
           nop
           nop
           nop
           nop
           nop
           nop
           mov   [di],ch                      (* Ausgaberegister Daten mit SCL high *)
           nop
           nop
           nop
           nop
           nop
           nop
           mov   [di+1],dl                    (* Ausgang ein -> 0 *)
           nop
           nop
           nop
           nop
           nop
           nop
           (* Der Bus hat nun die Startbedingung erkannt *)
           (* Mit der Ausgabe beginnen                   *)
@@nstart:  mov   [di],cl                      (* SCL auf Low *)
@@byt_lp:  mov   al,byte ptr es:[bx]          (* Byte lesen *)
           mov   si,8                         (* 8 Bits *)
@@ctrl1:   mov   [di+1],dl                    (* Ausgang ein -> 0 Bit *)
           shl   al,1                         (* Bit in Carry schieben *)
           jnc   @@ctrl1_0                    (* ist auch 0 *)
           mov   [di+1],dh                    (* Bit auf 1 *)
@@ctrl1_0: nop
           nop
           nop
           nop
           nop
           nop
           mov   [di],ch                      (* Clock auf 1 *)
           nop
           nop
           nop
           nop
           nop
           nop
           mov   [di],cl                      (* Clock auf 0 *)
           nop
           nop
           nop
           nop
           nop
           nop
           dec   si                           (* 1 Bit weiter *)
           jnz   @@ctrl1                      (* und weiter *)
           mov   [di+1],dl                    (* Datenbit auf 0 (ACK prepare) *)
           nop
           nop
           nop
           nop
           nop
           nop
           mov   [di],ch                      (* Clock auf 1 *)
           nop
           nop
           nop
           nop
           nop
           nop
           (* hier angekommen mu das EEPROM nun mit ACK besttigen *)
           (* wenn kein ACK vom Slave kommt ergibt sich automatisch ein STOP *)
           mov   [di+1],dh                    (* Datenbit auf 1 (ACK Freigabe) *)
           nop
           nop
           nop
           nop
           nop
           nop
           test  byte ptr [di],SDA_BIT        (* Prfen ob 1 oder 0 *)
           jnz   @@error                      (* Ein Fehler liegt vor -> Ende *)
           mov   [di],cl                      (* Clock auf 0 *)
           inc   bx                           (* ein Byte weiter *)
           dec   ah                           (* Anzahl - 1*)
           jnz   @@byt_lp                     (* und nochmal *)
           jmp   @@ende
@@error:   mov   @result,3                    (* Fehler Device nicht gefunden *)
@@ende:    push  ds
           pop   es                           (* es setzen *)
           pop   ds                           (* altes Datenregister *)
           mov   ah,[pm0_mirror]                (* Wert alter Modus *)
           and   ah,not SCL_BIT               (* Clock = Output *)
           mov   byte ptr es:[di+1],ah        (* setzen *)
           popf
  end;  (* end asm *)
(*$ENDIF *)

end;  (* out_iic *)

(*******************************************)
(* Einlesen vom dem IIC-Bus.               *)
(* anz = 0 entspricht 256 Byte !           +)
(* Es wird davon ausgegangen, da SCL auf  *)
(* Low ist.                                *)
(* Das letzte Byte wird nicht besttigt,   *)
(* sondern der Transfer wird dort mit einem*)
(* Stop/Abort abgebrochen !                *)
(* Registerbelegung:                       *)
(* ds    : Zeiger auf V25-IO-Segment       *)
(* di    : Zeiger Port P0 (Daten)          *)
(* ch,cl : High und Low Werte SCL          *)
(* dh,dl : High und Low Werte SDA          *)
(* es:bx : Zeiger auf Datenbytes           *)
(* ah    : Bytezhler                      *)
(* al    : temporre Bitdaten              *)
(* si    : Loopzhler                      *)
(*******************************************)
procedure in_iic(var buf;anz:byte);
begin (* in_iic *)
(*$IFnDEF SCC *)
  blockread(filePC,buf,anz);
{$ELSE}
  asm   (* wegen Geschwindigkeit wird die Operation in asm ausgefhrt *)
           les   bx,dword ptr [buf]           (* Zeiger Ausgabe *)
           pushf                              (* Interruptzustand *)
           cli                                (* retten + IRQ sperren *)
           mov   dh,[pm0_mirror]                (* Output Variable P0 Direction *)
           push  ds                           (* fr Registerzugriff retten *)
           mov   ax,seg p0                    (* Segment V25-IO *)
           mov   ds,ax                        (* nach ds *)
           and   dh,not SCL_BIT               (* SCL immer Output *)
           or    dh,SDA_BIT                   (* SDA als Input *)
           mov   dl,dh                        (* nach dl *)
           and   dl,not SDA_BIT               (* Ausgang ein SDA = 0 *)
           mov   di,offset P0                 (* Zeiger Daten 0 *)
           mov   ch,[di]                      (* aktuellen Zustand lesen *)
           and   ch,not SDA_BIT               (* Data auf 0 *)
           or    ch,SCL_BIT                   (* Clock auf 1 *)
           mov   cl,ch                        (* Clock Low *)
           and   cl,not SCL_BIT               (* setzen *)
           mov   ah,[anz]                     (* Anzahl *)
@@byt_lp:  mov   si,8                         (* 8 Bits *)
           mov   [di+1],dh                    (* Ausgang wird High *)
@@ctrl1:   nop
           nop
           nop
           nop
           nop
           nop
           mov   [di],ch                      (* Clock auf 1 *)
           nop
           nop
           nop
           nop
           nop
           nop
           nop
           nop
           nop
           test  byte ptr [di],SDA_BIT        (* Test ob Bit High *)
           jz    @@is0                        (* nein ist 0 *)
           stc                                (* Carry set *)
           jmp   @@is1                        (* ist eine 1 *)
@@is0:     clc                                (* Carry lschen *)
@@is1:     rcl   al,1                         (* Carry in Bit schieben *)
           mov   [di],cl                      (* Clock auf 0 *)
           dec   si                           (* 1 Bit weiter *)
           jnz   @@ctrl1                      (* und weiter *)
           mov   byte ptr es:[bx],al          (* ablegen *)
           inc   bx                           (* ein Byte weiter *)
           mov   [di+1],dl                    (* Datenbit auf 0 (Ack) *)
           nop
           nop
           nop
           nop
           nop
           nop
           mov   [di],ch                      (* Clock auf 1 *)
           dec   ah                           (* Ende ? *)
           jz    @@ende                       (* ja *)
           mov   [di],cl                      (* Clock auf 0 *)
           nop
           nop
           nop
           nop
           nop
           nop
           mov   [di+1],dh                    (* Daten auf 1 *)
           nop
           nop
           nop
           nop
           jmp   @@byt_lp                     (* und weiter *)
@@ende:    mov   [di+1],dh                    (* Daten auf 1 (Abort/Stop) *)
           pop   ds                           (* altes Datenregister *)
           popf
  end;  (* end asm *)
(*$ENDIF *)
end;  (* in_iic *)

(******************************************)
(* Stopbedingung auf dem IIC-Bus erzeugen *)
(******************************************)
procedure stop_iic;
begin (* stop_iic *)
(*$IFnDEF SCC *)
  exit;
(*$ELSE *)
  asm
    mov   ax,seg p0                    (* Segment V25-IO *)
    mov   es,ax                        (* nach es *)
    mov   si,offset p0                 (* Zeiger IO-Port *)
    pushf                              (* Interruptzustand *)
    cli                                (* retten + IRQ sperren *)
    mov   dh,[pm0_mirror]                (* Output Variable P0 Direction *)
    and   dh,not SCL_BIT               (* Clock = Output *)
    and   dh,not SDA_BIT               (* Output Daten = 0 *)
    mov   byte ptr es:[si+1],dh        (* SDA auf 0, SCL auf Ausgang *)
           nop
           nop
           nop
           nop
           nop
           nop
           nop
           nop
           nop
    nop
    nop
    nop
    mov   dl,byte ptr es:[si]          (* aktuelle Daten lesen *)
    or    dl,SCL_BIT                   (* Clock auf 1 *)
    and   dl,not SDA_BIT               (* Daten auf 0 *)
    mov   byte ptr es:[si],dl          (* Clock anheben *)
           nop
           nop
           nop
           nop
    nop
    nop
    or    dh,SDA_BIT                   (* SDA auf 1 *)
    mov   byte ptr es:[si+1],dh        (* SCL setzen *)
    mov   [pm0_mirror],dh                (* und so speichern *)
           nop
           nop
           nop
           nop
           nop
           nop
    nop
    nop
    nop
    popf
  end;
(*$ENDIF *)
end;  (* stop_iic *)

(*********************************************)
(* In den Cache schreiben und programmieren. *)
(* Keine Prfung auf Overscan.               *)
(*********************************************)
function cache_prog(var buf;adr:word;anz:byte):integer;
var
  o      : out_type;
  result : integer;
  res1   : integer;
begin (* cache_prog *)
  o.cmd_bus := eeadr; (* Adresse Write *)
  o.adresse := swap(adr); (* Anfangsadresse *)
(*$IFnDEF SCC *)
  seek(filePC,swap(o.adresse));
  blockwrite(filePC,buf,anz);
  result := 0;
(*$ELSE *)
  result := out_iic(o,3,FALSE,FALSE);
  if result = 0 then
     result := out_iic(buf,anz,TRUE,TRUE); (* schreiben *)
  stop_iic; (* Programming Start *)
  if result = 0 then
   repeat
     res1 := out_iic(o,1,TRUE,FALSE);
     stop_iic; (* Bus Ende *)
   until res1 = 0;
(*$ENDIF *)
  cache_prog := result;
end;  (* cache_prog *)

(*********************)
(* Einlesen aus File *)
(*********************)
function InputFunc(var eefile:text):integer; far;
var
  result    : integer;
  soll_read : word;
  o         : out_type;
begin (* InputFunc *)
  with textrec(eefile) do
   begin (* with *)
     result := 0; (* falls kein Fehler *)
     if (not read_eof) and (BufPos >= BufEnd) then
      begin (* weiter lesen *)
        BufPos := 0; (* An Anfang *)
        BufEnd := 0; (* erstmal nix gelesen *)
        if bufferakt + BufSize > 8192 then soll_read := 8192 - bufferakt (* begrenzen *)
                                      else soll_read := BufSize;
        o.cmd_bus := eeadr; (* Adresse Write *)
        o.adresse := swap(bufferakt); (* Anfangsadresse *)
        result    := out_iic(o,3,FALSE,FALSE); (* Kommando starten *)
        stop_iic;
        inc(o.cmd_bus,1); (* Read Bit setzen *)
        if result = 0 then result := out_iic(o,1,FALSE,FALSE); (* starten *)
        if result = 0 then in_iic(Buffer,soll_read);
        if result = 0 then
          while (BufEnd < BufSize) and (Buffer[BufEnd] <> #26)
            do begin
               inc(BufEnd);
               inc(bufferakt);
               end;
        if BufEnd = 0 then read_eof := TRUE;
      end;  (* weiter lesen *)
   end;  (* with *)
  InputFunc := result;
end;  (* InputFunc *)

(*********************)
(* Schreiben in File *)
(*********************)
function OutputFunc(var eefile:text):integer; far;
var
  result : integer;
  w      : word;
  p      : ^byte;
begin (* OutputFunc *)
  with textrec(eefile) do
    begin (* with *)
    result := 0; (* falls kein Fehler *)
    if bufferakt + BufPos <= EEPROM_SIZE - 2 then
       begin (* weiter schreiben *)
       p := Pointer(@Buffer);
       for w := 1 to BufPos do
         begin
         buffpruef := buffpruef xor p^; (* Checksumme *)
         inc(word(p));
         end;
       p := Pointer(@Buffer); (* Zeiger auf Puffer *)
       if BufPos <> 0 then
         repeat
           w := 64 - (bufferakt and 7); (* Maximum erlaubt *)
           if BufPos < w then w := BufPos; (* Falls weniger *)
           result := cache_prog(p^,bufferakt,w);
           inc(word(p),w); (* Zeiger erhhen *)
           inc(bufferakt,w); (* Adresse auch *)
           dec(BufPos,w); (* Zhler erniedrigen *)
         until (BufPos = 0) or (result <> 0);
      end   (* weiter schreiben *)
     else result := 29; (* Diskette voll *)
     BufPos := 0;
   end;  (* with *)
  OutputFunc := result;
end;  (* OutputFunc *)

(************************)
(* Schliessen des Files *)
(************************)
function CloseFunction(var eefile:text):integer; far;
var
  result       : integer;
  p            : ^byte;
  w            : word;
begin (* CloseFunction *)
  result := 0; (* kein Fehler *)
  with textrec(eefile) do
   begin (* with *)
     if mode = fmOutput then
      begin (* Puffer leeren *)
        Buffer[0] := #26; (* 26 *)
        Buffer[1] := chr(buffpruef xor 26); (* Prfsumme *)
        BufPos    := 2;
        result    := OutputFunc(eefile); (* Abschlu schreiben *)
      end;  (* Puffer leeren *)
     mode   := fmClosed;
     handle := $FFFF;
(*$IFnDEF SCC *)
     close(filePC);
(*$ENDIF *)
     eeprom_inuse := FALSE; (* freigeben *)
   end;  (* with *)
  CloseFunction := result;
end;  (* CloseFunction *)

(*********************************************************)
(* Open24C65 ffnet das EEPROM zum Lesen oder Schreiben. *)
(*********************************************************)
function Open24C65(var eefile:text):integer; far;
var
  o : out_type;

  function Check24c65:boolean; near;
  var
    result   : integer;
    w,w1     : word;
    p        : ^byte;
    b        : byte;
    pruef    : byte;
    ok       : boolean;
    eof_flag : boolean;

  begin (* Check24C65 *)
    Check24C65 := FALSE; (* falls Fehler *)
    (* erstmal die Adresse des EEPROMs feststellen *)
    eeadr := $A0 - 2 ; (* tiefste Adresse - Increment *)
    repeat
      inc(eeadr,2); (* Adresse einen Baustein erhhen *)
      o.cmd_bus  := eeadr; (* Kommando setzen *)
      o.adresse  := 0;     (* Adresse 0 *)
      result := out_iic(o,3,FALSE,FALSE); (* Prfen ob vorhanden *)
      stop_iic;
    until (result = 0) or ((eeadr and $0F) = $0E);
    if result = 0 then
     begin (* EEPROM gefunden *)
(*$IFnDEF SCC *)
       reset(filePC,1);
(*$ENDIF *)
       Check24C65 := TRUE; (* EEPROM da *)
       if textrec(eefile).mode = fmInput then
        with textrec(eefile) do
         begin (* Prfsumme usw. *)
           w := 0; (* Anfang des EEPROMs *)
           ok := FALSE; (* kein Abbruch *)
           pruef := $AA;
           eof_flag := FALSE;
           o.cmd_bus := o.cmd_bus or 1; (* Read Bit setzen *)
           repeat
             if result = 0 then result := out_iic(o,1,FALSE,FALSE); (* Current Read *)
             if result = 0 then in_iic(Buffer,sizeof(Buffer));
             watchdog;
             p := Pointer(@Buffer);
             for w1 := 1 to sizeof(buffer) do
             if (not ok) and (result = 0) then
               begin
               b := p^;
               watchdog;
               pruef := pruef xor b; (* Prfsumme Check *)
               watchdog;
               ok := eof_flag or (w + w1 >= EEPROM_SIZE);
               inc(word(p)); (* Zeiger weiter *)
               if b = 26 then eof_flag := TRUE;
               end;
             inc(w,sizeof(Buffer));
           until ok or (result <> 0);
            ok := TRUE;
           read_eof := (not eof_flag) or (pruef <> 0);
         end;  (* Prfsumme usw. *)
     end;  (* EEPROM gefunden *)
    if result <> 0 then Check24c65 := FALSE;
  end;  (* Check24C65 *)

begin (* Open24C65 *)
  with textrec(eefile) do
    begin (* with *)
    if eeprom_inuse then
      begin
      Open24C65 := 32; (* Gleichzeitiger Zugriff *)
      exit;
      end;
    if (mode <> fmInput) and (mode <> fmOutput) then
      begin
      Open24C65 := 12; (* Ungltiger Zugriffscode *)
      exit;
      end;
    if mode = fmInput then InOutFunc := @InputFunc
                      else InOutFunc := @OutPutFunc;
    FlushFunc := InOutFunc;
    (* Testen on EEPROM vorhanden *)
    Open24C65 := 3; (* falls nicht vorhanden *)
    if not Check24c65 then exit; (* falls nicht vorhanden *)
    Open24C65 := 0; (* vorhanden *)
    closefunc := @closefunction;
    BufSize   := sizeof(Buffer);
    BufPos    := 0;
    BufEnd    := 0;
    BufPtr    := @Buffer;
    Handle    := 10;
    Bufferakt := 0;
    buffpruef := $AA; (* Init Prfsumme *)
    eeprom_inuse := TRUE; (* EEPROM wird gerade benutzt *)
    end;  (* with *)
end;  (* Open24C65 *)

(************************************************************************)
(* Funktion zum Setzen eines Standard-Textfiles fr Benutzung mit       *)
(* dem eeprom. Diese Prozedur mu auf jeden Fall VOR dem Aufruf eines   *)
(* Rewrite oder Resets durchgefhrt werden.                             *)
(************************************************************************)
procedure Assign24c65(var eefile:text);
begin (* assign24c65 *)
  with textrec(eefile) do
    begin
    Name[0]   := #0; (* Name leer *)
    handle    := $FFFF; (* ungltig *)
    mode      := fmClosed; (* File ist zu *)
    OpenFunc  := @Open24C65; (* Zeiger auf Open Funktion *)
    end;
end;  (* assign24c65 *)


begin (* Unit fd_24c65 *)
  WriteLn ('- 24c65-Init');
  eeprom_inuse := FALSE; (* EEPROM aktuell nicht benutzt *)
(*$IFnDEF SCC *)
  if cseg > $A000 then halt(255);
  Assign(filePC,'24c65.dat');
{  rewrite(filePC,1);
  blockwrite(filePC,filePC,EEPROM_SIZE);
  close(filePC); }
(*$ELSE *)
  IF cseg < $A000 then halt(255);
  pm0_mirror := (pm0_mirror and (not SCL_BIT)) or SDA_BIT;
  PM0_reg    := pm0_mirror;
  p0         := (p0 and (not SDA_BIT)) or SCL_BIT;
(*$ENDIF *)
end.  (* Unit fd_24c65 *)
